package org.seanmonahan.effects
{
	//=====================================
	// Import statements
	//=====================================
	
	import flash.display.DisplayObject;
	
	import mx.effects.easing.Linear;
	import mx.effects.Move;
	import mx.effects.IEffectInstance;
	import mx.core.UIComponent;
	import mx.events.EffectEvent;
	import mx.effects.Effect;
	
	/**
	 * A shake effect.
	 */
	public class Shake extends Effect
	{
		//=====================================
		// Vars
		//=====================================
		
		/**
		 * The move effect we will manipulate to achieve a shake effect.
		 * 
		 * @private
		 */
		private var _mover:Move = new Move();
		
		/**
		 * Number of times to shake.
		 * 
		 * @private
		 */		
		private var _numShakes:int;
		
		/**
		 * Counter to track the current shake we are on.
		 * 
		 * @private
		 */
		private var _currentShake:int;
		
		/**
		 * Distance the target should shake in the X direction.
		 * 
		 * @private
		 */
		private var _distX:int;
		
		/**
		 * Distance the target should shake in the Y direction.
		 * 
		 * @private
		 */
		private var _distY:int;
		
		/**
		 * Flag used to indicate if the effect should automatically reset 
		 * when effect finishes.
		 * 
		 * @private
		 */
		private var _resetOnEnd:Boolean;
		
		/**
		 * The direction the effect should start shaking along the X axis.
		 * E.g., right (1) or left (-1).
		 * 
		 * @private
		 */
		private var _shakeDirectionX:int;
		
		/**
		 * The direction the effect should start shaking along the Y axis.
		 * E.g., down (1) or up (-1).
		 * 
		 * @private
		 */
		private var _shakeDirectionY:int;
		
		/**
		 * Flag indicating the effect can play.
		 * This is used prevent the effect from playing if it has
		 * not been reset.
		 * 
		 * @private
		 * @default true
		 */
		private var _canPlay:Boolean = true;
		
		//=====================================
		// Constructor
		//=====================================
		
		/**
		 * Constructor
		 */
		public function Shake()
		{
			// create a new Move effect and set it to use linear easing
			_mover.easingFunction = mx.effects.easing.Linear.easeIn;
			
			// Listen to EFFECT_END events on our mover
			_mover.addEventListener(EffectEvent.EFFECT_END, _handleEffectEnd);
			
			// Set some default values.
			_numShakes = 5;
			_distX = 25;
			_distY = 0;
			_currentShake = 1;
			_resetOnEnd = true;
			_shakeDirectionX = 1;
			_shakeDirectionY = 1;
		}
		
		//=====================================
		// Public methods
		//=====================================
		
		/**
		 * Interrupts the effect that is currently playing and jumps immediately 
		 * to the end of the effect.
		 * 
		 * @param effectInstance Not used.  Required for override.
		 */
		public override function end(effectInstance:IEffectInstance = null):void
		{
			_mover.end();
		}
		
		/**
		 * Begins playing the effect.
		 * 
		 * @param targets Not used.  Required for override.
		 * @param playReversedFromEnd Not used.  Required for override.
		 * @return Empty Array.  Required for override.
		 */
		public override function play(targets:Array = null, playReversedFromEnd:Boolean = false):Array
		{
			if (_canPlay)
				_mover.play();
				
			return new Array();
		}
		
		/**
		 * Pauses the effect until resume() is called.
		 */
		public override function pause():void
		{
			_mover.pause();
		}
		
		/**
		 * Resumes the effect after a call to pause().
		 */
		public override function resume():void
		{
			_mover.resume();
		}
		
		/**
		 * Resets the effect to it's initial settings.
		 */
		public function reset():void
		{
			_currentShake = 1;
			_canPlay = true;
		}
		
		//=====================================
		// Private and protected methods
		//=====================================
		
		/**
		 * Event handler for the effect's EFFECT_END event.
		 * 
		 * @param event The EffectEvent dispatched when the effect ends.
		 */
		protected function _handleEffectEnd(event:EffectEvent):void
		{
			// increment our shake count
			_currentShake++;
			
			if (_currentShake < _numShakes)
			{
				// If we have not reached the shake limit
				if (_currentShake % 2 == 0)
				{
					// An even _currentShake means we need to move left.
					_mover.xBy = (_distX * -1) * _shakeDirectionX;
					_mover.yBy = (_distY * -1) * _shakeDirectionY;
				}
				else
				{
					// An odd _currentShake means we need to move right
					_mover.xBy = _distX * _shakeDirectionX;
					_mover.yBy = _distY * _shakeDirectionY;
				}
				play();
			}
			else if (_currentShake == _numShakes)
			{
				// This is our last shake.
				
				if (_currentShake % 2 == 0)
				{
					_mover.xBy = Math.ceil(_distX * 0.5) * -1 * _shakeDirectionX;
					_mover.yBy = Math.ceil(_distY * 0.5) * -1 * _shakeDirectionY;
				}
				else
				{
					_mover.xBy = Math.ceil(_distX * 0.5) * _shakeDirectionX;
					_mover.yBy = Math.ceil(_distY * 0.5) * _shakeDirectionY;
				}
				
				play();
			}
			else
			{
				// We've done all our shaking.
				
				// Dispatch the EFFECT_END event
				this.dispatchEvent(new EffectEvent(EffectEvent.EFFECT_END));
				
				// Reset the movement params to their initial values
				_mover.xBy = Math.ceil(_distX * 0.5) * _shakeDirectionX;
				_mover.yBy = Math.ceil(_distY * 0.5) * _shakeDirectionY;
				
				if(_resetOnEnd)
				{
					// Reset if our flag is set
					_canPlay = true;
					reset();
				}
				else
				{
					// Otherwise don't.
					_canPlay = false;
					end();
				}
			}
			
		}
		
		//=====================================
		// Getters and setters
		//=====================================
		
		/**
		 * The target for the effect.
		 * 
		 * @return The target of the effect.
		 */
		public override function get target():Object
		{
			return _mover.target;
		}
		
		/**
		 * The target for the effect.
		 * 
		 * @param t The effect target.
		 */
		public override function set target(t:Object):void
		{
			var c:UIComponent = UIComponent(t);
			_mover.target = t;
		}
		
		/**
		 * The distance the effect should shake along the
		 * x-axis.
		 * 
		 * @param d The distance in pixels.
		 */
		public function set distX(d:Number):void
		{
			_distX = d;
			_mover.xBy = Math.ceil(_distX * 0.5) * _shakeDirectionX;
		}
		
		/**
		 * The distance the effect should shake along the
		 * y-axis.
		 * 
		 * @param d The distance in pixels.
		 */
		public function set distY(d:Number):void
		{
			_distY = d;
			_mover.yBy = Math.ceil(_distY * 0.5) * _shakeDirectionY;
		}
		
		/**
		 * The easing function to be used for shaking.
		 * 
		 * @param f The easing function.
		 */
		public function set easingFunction(f:Function):void
		{
			_mover.easingFunction = f;
		}
		
		/**
		 * The duration of the effect in milliseconds.
		 * 
		 * @param d The duration in milliseconds.
		 */
		public override function set duration(d:Number):void
		{
			_mover.duration = d / _numShakes;
		}
		
		/**
		 * The number of times the effect should shake.
		 * 
		 * @param n The number of shakes.
		 */
		public function set numShakes(n:int):void
		{
			_numShakes = n;
		}
		
		/**
		 * Flag indicating the effect should automatically 
		 * reset when completed.
		 * 
		 * True means the effect will automatically reset.
		 * False means no automatic reset.
		 * 
		 * @param v The value for the flag.
		 */
		public function set resetOnEnd(v:Boolean):void
		{
			_resetOnEnd = v;
		}
		
		/**
		 * The direction the effect should start shaking along the X axis.
		 * E.g., right or left.
		 * 
		 * @param d The direction.
		 */
		[Inspectable(enumeration="right,left", defaultValue="right")]
		public function set shakeDirectionX(d:String):void
		{
			switch (d)
			{
				case "left":
					_shakeDirectionX = -1;
					break;
				case "right":
				default:
					_shakeDirectionX = 1;
					break;
			}
			_mover.xBy = Math.ceil(_distX * 0.5) * _shakeDirectionX;
		}
		
		/**
		 * The direction the effect should start shaking along the Y axis.
		 * E.g., down or up.
		 * 
		 * @param d The direction.
		 */
		[Inspectable(enumeration="up,down", defaultValue="down")]
		public function set shakeDirectionY(d:String):void
		{
			switch (d)
			{
				case "up":
					_shakeDirectionY = -1;
					break;
				case "down":
				default:
					_shakeDirectionY = 1;
					break;
			}
			_mover.yBy = Math.ceil(_distY * 0.5) * _shakeDirectionY;
		}
		
	}
}